import React, { useState } from 'react';
import {
    Table,
    TableBody,
    TableCell,
    TableContainer,
    TableHead,
    TableRow,
    Paper,
    TablePagination,
    Rating,
    Chip,
    MenuItem,
    FormControl,
    Select,
    InputLabel,
    Box,
} from '@mui/material';
import { Link } from 'react-router-dom';
import { Article } from '../types/article';
import { capitalizeCzech, mapToName, statusColor } from '../utils/cz_uk';

const STATUSES = ['Schváleno', 'Čeká na schválení', 'Zamítnuto'];

interface ArticleTableProps {
    articles: Article[];
}

const ArticleTable = (props: ArticleTableProps) => {
    const { articles } = props;
    const [page, setPage] = useState(0);
    const [statusFilter, setStatusFilter] = useState('');
    const rowsPerPage = 20;

    const handleChangePage = (event: unknown, newPage: number) => setPage(newPage);
    const handleFilterChange = (event: React.ChangeEvent<{ value: unknown }>) => {
        setStatusFilter(event.target.value as string);
        setPage(0);
    };

    

    const filteredRows = statusFilter
        ? articles.filter((row) => mapToName(row.status) === statusFilter)
        : articles;

    return (
        <Paper className="shadow-lg p-4">
            <Box mb={2} display="flex" justifyContent="flex-end">
                <FormControl size="small" sx={{ minWidth: 200 }}>
                    <InputLabel>Filtrovat stav</InputLabel>
                    <Select
                        value={statusFilter}
                        onChange={handleFilterChange}
                        label="Filtrovat stav"
                    >
                        <MenuItem value="">Všechny stavy</MenuItem>
                        {STATUSES.map((status) => (
                            <MenuItem key={status} value={status}>
                                {status}
                            </MenuItem>
                        ))}
                    </Select>
                </FormControl>
            </Box>

            <TableContainer>
                <Table>
                    <TableHead>
                        <TableRow className="bg-yellow-200">
                            <TableCell>
                                <strong>Název článku</strong>
                            </TableCell>
                            <TableCell>
                                <strong>Téma</strong>
                            </TableCell>
                            <TableCell align="center">
                                <strong>Stav</strong>
                            </TableCell>
                            <TableCell align="center">
                                <strong>Hodnocení</strong>
                            </TableCell>
                        </TableRow>
                    </TableHead>
                    <TableBody>
                        {filteredRows
                            .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                            .map((row) => (
                                <TableRow key={row.id} className="hover:bg-yellow-100">
                                    <TableCell>
                                        <Link
                                            to={`/article_preview/${row.id}`}
                                            className="text-green-950 font-medium hover:underline"
                                        >
                                            {row.title}
                                        </Link>
                                    </TableCell>
                                    <TableCell>{capitalizeCzech(row.topic)}</TableCell>
                                    <TableCell align="center">
                                        <Chip
                                            label={mapToName(row.status)}
                                            color={statusColor(row.status)}
                                            sx={{ minWidth: 150, textAlign: 'center' }}
                                        />
                                    </TableCell>
                                    <TableCell align="center">
                                        <Rating value={row.rating} readOnly />
                                    </TableCell>
                                </TableRow>
                            ))}
                    </TableBody>
                </Table>
            </TableContainer>

            <TablePagination
                component="div"
                count={filteredRows.length}
                rowsPerPage={rowsPerPage}
                page={page}
                onPageChange={handleChangePage}
                rowsPerPageOptions={[rowsPerPage]}
                className="bg-yellow-200"
                labelDisplayedRows={({ from, to, count }) =>
                    `${from}–${to} z ${count !== -1 ? count : `více než ${to}`}`
                }
            />
        </Paper>
    );
};

export default ArticleTable;
