// src/pages/LoginPage.tsx
import React, { useState, FormEvent } from 'react';
import { useNavigate } from 'react-router-dom';
import { useUser } from '../contexts/UserContext';
import { User } from '../types/user';
import api from '../utils/api';                // centralized Axios instance


const LoginPage: React.FC = () => {
    const navigate = useNavigate();
    const [username, setUsername] = useState('');
    const [password, setPassword] = useState('');
    const [error, setError] = useState('');

    const { loginWithCredentials } = useUser();

    const handleSubmit = async (e: FormEvent) => {
        e.preventDefault();
        setError(''); // Clear previous error

        try {
            await loginWithCredentials({ username, password });
            navigate('/');
        }
        catch (err: any) {
            if (err.response?.data?.message) {
                setError(err.response.data.message);
            } else if (err.response?.status === 401) {
                setError('Špatné uživatelské jméno nebo heslo');
            } else {
                console.error('Login error:', err);
                setError('Vyskytla se neočekávaná chyba při přihlašování');
            }
        }
    };

    return (
        <div
            className="min-h-screen p-8 flex items-center justify-center text-center bg-gray-100"
            style={{
                backgroundImage:
                    "url('https://www.collegetransitions.com/wp-content/uploads/2024/03/math-formulas.jpg')",
                backgroundSize: 'cover',
                backgroundPosition: 'center',
            }}
        >
            <form
                onSubmit={handleSubmit}
                className="flex flex-col w-full gap-4 max-w-sm mx-auto p-8 bg-white rounded-lg shadow-lg"
            >
                <h2 className="font-bold mb-6 text-green-950 drop-shadow-lg">
                    Login
                </h2>

                <input
                    className="p-2 border border-gray-300 rounded"
                    type="text"
                    placeholder="Uživatelské jméno"
                    value={username}
                    onChange={e => setUsername(e.target.value)}
                />

                <input
                    className="p-2 border border-gray-300 rounded"
                    type="password"
                    placeholder="Heslo"
                    value={password}
                    onChange={e => setPassword(e.target.value)}
                />

                <button
                    type="submit"
                    className="p-2 bg-green-900 text-white rounded hover:bg-green-800 transition"
                >
                    Login
                </button>

                <button
                    type="button"
                    className="text-green-950 hover:underline"
                    onClick={() => navigate('/register')}
                >
                    Vytvořit účet
                </button>

                {error && <p className="mt-4 text-red-600">{error}</p>}
            </form>
        </div>
    );
};

export default LoginPage;
